// Buffers.c
#include "buffers.h"
#include <stdlib.h>
#include <string.h>

ring_buffer *create_ring_buffer(int size){
  // Allocate all memory in one operation
  ring_buffer *buffer;

  buffer = (ring_buffer*) malloc(sizeof(ring_buffer)+size); // Ring buffer for incomming data
  
  if(buffer == NULL)
    return NULL;

  // Initialize the ring buffer
  buffer->size = size;
  buffer->start_offset = 0;
  buffer->stop_offset = 0;
  buffer->data = (unsigned char*) buffer + sizeof(ring_buffer);//malloc(size);   
  // printf("Buffer:      %d\n", (int) buffer); 
  // printf("Data:        %d\n", (int) buffer->data); 
  // printf("Data-buffer: %d\n", (int) buffer->data - (int) buffer); 
  // printf("sizeof:      %d\n", sizeof(ring_buffer)); 

  return buffer;
}    

int destroy_ring_buffer(ring_buffer *buffer){
  //free(buffer->data);
  free(buffer);
  return 0;
}

// ============================= Ring buffer functions ================================

// This function returns the amount of data in a ring_buffer in bytes
// If the buffer is full this function returns -1
int bytes_in_buffer(ring_buffer *buffer){
  // If the buffer is full
  if(buffer->stop_offset == -1){
    return buffer->size;
  }
  // If the buffer is empty 
  if(buffer->start_offset == buffer->stop_offset){
    return 0;
  }
  // If there is data in the buffer and NO boundary crossing
  if(buffer->start_offset < buffer->stop_offset){
    return (buffer->stop_offset - buffer->start_offset);
  }
  
  // If there is data in the buffer AND boundary crossing
  if(buffer->start_offset > buffer->stop_offset){
    return ((buffer->size - buffer->start_offset) + buffer->stop_offset);
  }

  return 0;
}

int free_cont_space(ring_buffer *buffer){
  // If the buffer is full
  if(buffer->stop_offset == -1)
    return 0;

  // If there is free space in the buffer and NO boundary crossing (on the free space :-) )
  if(buffer->start_offset > buffer->stop_offset)
    return buffer->start_offset - buffer->stop_offset;

  // If there is free space in the buffer AND maybe boundary crossing
  if(buffer->start_offset <= buffer->stop_offset)
    return buffer->size - buffer->stop_offset;

  return 0;
}

char putbyte(ring_buffer *buffer, unsigned char byte){
  // Check for buffer full
  if(buffer->stop_offset == -1)
	   return -1;
  
  // Insert the byte
  buffer->data[buffer->stop_offset] = byte;

  //Update stop_offset
  buffer->stop_offset = (buffer->stop_offset+1)&buffer->mask;

  // If the buffer is full after the operation (no more free bytes)
  if(buffer->stop_offset == buffer->start_offset)
    buffer->stop_offset = -1;

  return 0;
}

char getbyte(ring_buffer *buffer, unsigned char *byte){
  // Check for available data in buffer (Errorcode -1)
  if(bytes_in_buffer(buffer) == 0)
    return -1;

  *byte = buffer->data[buffer->start_offset];

  // If the buffer was full set stop_offset to start_offset to indicate free space
  if(bytes_in_buffer(buffer) == buffer->size)
    buffer->stop_offset = buffer->start_offset;      
  buffer->start_offset++;

  if(buffer->start_offset == buffer->size)
    buffer->start_offset = 0;

  return 0;
}

// Returns the value of a byte at a given location in the buffer
char peekbyte(ring_buffer *buffer, unsigned char *byte, int offset){
  int bytes_in_buf;

  bytes_in_buf = bytes_in_buffer(buffer);
  
  // Check for available data in buffer (Errorcode -1)
  if(bytes_in_buf == 0)
    return -1;

  // Return 2 if the offset is larger than the amount of data in the buffer
  if(offset >= bytes_in_buf)
    return 2;

  if(offset < buffer->size - buffer->start_offset)
    //No zero crossing
    *byte = buffer->data[buffer->start_offset + offset];
  else
    //With zero crossing
    *byte = buffer->data[buffer->start_offset + offset - buffer->size];

  return 0;
}

int find_char(ring_buffer *buffer, int search_start_offset, unsigned char lookfor){
  int bytes_in_buf, cont_data, i;

  //Get the number of bytes in the buffer
  bytes_in_buf = bytes_in_buffer(buffer);

  //Return if there is no data to search through
  if(bytes_in_buf<1)
    return -1;

  //If there is no boundary crossing on data in buffer
  if(buffer->stop_offset>buffer->start_offset){
    //Ensure that search_from_offset is in data and not in free space
    if((search_start_offset < buffer->start_offset) || (buffer->stop_offset<=search_start_offset))
      return -2;

    //Find how much continuous data is available following the search_start_offset
    cont_data = buffer->stop_offset - search_start_offset;

    //Search through the continuous data
    for(i=0;i<cont_data;i++)
      if(*(buffer->data+search_start_offset+i)==lookfor)
        return (int)(search_start_offset+i);
  }
  //If there is boundary crossing on data in buffer
  else{
    //Ensure that search_from_offset is in data and not in free space
    if((search_start_offset<buffer->start_offset) && (buffer->stop_offset<=search_start_offset))
      return -3;

    //If there is no boundary crossing on the data that should be searched
    if(buffer->stop_offset>search_start_offset){
      //Find how much continuous data is available following the search_start_offset
      cont_data = buffer->stop_offset - search_start_offset;

    //Search through the continuous data
    for(i=0;i<cont_data;i++)
      if(*(buffer->data+search_start_offset+i)==lookfor)
        return (int)(search_start_offset+i);
    }
    //If there is boundary crossing on the data that should be searched
    else{
      //Find how much continuous data is available following the search_start_offset
      cont_data = buffer->size - search_start_offset;

      //Search through the continuous data
      for(i=0;i<cont_data;i++)
        if(*(buffer->data+search_start_offset+i)==lookfor)
          return (int)(search_start_offset+i);
      
    //Search the rest
    for(i=0;i<buffer->stop_offset;i++)
      if(*(buffer->data+i)==lookfor)
        return (int)i;
    }
  }

  //Return with error if character was not found
  return -4;
}

char search_buffer(ring_buffer *buffer, unsigned char *lookfor, int lookfor_length, int start_offset, int *found_offset){
  int bytes_in_buf = 0;
  int count = 0, count2 = 0, count3 = 1;
  char error = 0;
  unsigned char peek_ch;

  bytes_in_buf = bytes_in_buffer(buffer);
  if(lookfor_length > bytes_in_buf)
    return -1;

  // Look for the first char of lookfor in the buffer
  for(count = start_offset;count<bytes_in_buf - lookfor_length + 1;count++){
    peekbyte(buffer, &peek_ch, count);
    *found_offset = count;
    if((char) (peek_ch&255) == (char) (*lookfor&255)){
      
      error = 0;
      count3 = 1;
            
      for(count2=count+1;count2<lookfor_length + count;count2++){
        peekbyte(buffer, &peek_ch, count2);
	
        if((char) peek_ch != (char) *(lookfor+count3)){
          error = 1;
          break;
        }
        count3++;
      }

      //Check if all bytes matched
      if(error==0){
        return 0;
      }
    }
  }
  
  // Return 1 if lookfor was not found
  return -1;
}

char buf_write(ring_buffer *buffer, unsigned char *data, int numbytes){
  int free_space, i;

  //If this wasn't here and the buffer was empty the stop_offset would be set
  //to indicate full even though no data was written and then the buffer
  //go from empty to full - with no data written
  if(numbytes==0)
    return -1;

  // Return if there is not sufficient space in buffer
  if((buffer->size - bytes_in_buffer(buffer)) < numbytes)
    return -1;

  //Find how much free continuous space is available following stop_offset
  free_space = free_cont_space(buffer);

  //If there is boundary crossing
  if(numbytes > free_space){
    //Fill the free continuous space
    for(i=0;i<free_space;i++)
      *(buffer->data + buffer->stop_offset + i) = *(data+i);
    //Cross boundaries
    buffer->stop_offset = 0;
    //Fill in the rest of the data
    for(i=0;i<numbytes-free_space;i++)
      *(buffer->data + buffer->stop_offset + i) = *(data+i+free_space);
  }
  //If there is no boundary crossing
  else
    //Fill the data into the free continuous space
    for(i=0;i<numbytes;i++)
      *(buffer->data + buffer->stop_offset + i) = *(data+i);

  //Update stop_offset
  buffer->stop_offset = (buffer->stop_offset+i)&buffer->mask;

  //Check if buffer is has been filled
  if(buffer->stop_offset==buffer->start_offset)
    buffer->stop_offset=-1;

  return 0;
}

char buf_read(ring_buffer *buffer, unsigned char *data, int numbytes){
  int i, cont_data;

  //If this wasn't here and the buffer was full the stop_offset would be set
  //to indicate free space even though no data was read and then the buffer
  //would go from full to empty - with no data read
  if(numbytes==0)
    return 0;

  //Return with error if we are attempting to read more data than is available
  if(numbytes > bytes_in_buffer(buffer))
    return -1;

  //If the buffer was full update the stop_offset
  //buffer->stop_offset = buffer->start_offset;

  //Find how much continuous data is available following start_offset
  cont_data = buffer->size - buffer->start_offset;

  //If there is boundary crossing
  if(numbytes > cont_data){
    //Get the continuous data
    for(i=0;i<cont_data;i++)
      *(data+i) = *(buffer->data + buffer->start_offset + i);
    //Get the rest
    for(i=0;i<numbytes-cont_data;i++)
      *(data+cont_data+i) = *(buffer->data + i);
    //Update start_offset
    buffer->start_offset = i;
  }
  //If there is no boundary crossing
  else{
    //Get the data
    for(i=0;i<numbytes;i++)
      *(data+i) = *(buffer->data + buffer->start_offset + i);
    //Update the start_offset
    buffer->start_offset += i;
    if(buffer->start_offset == buffer->size)
      buffer->start_offset = 0;
  }

  return 0;
}

char remove_garbage(ring_buffer *buffer, int numbytes){
  //int i, cont_data;
  
  //If this wasn't here and the buffer was full the stop_offset would be set
  //to indicate free space even though no data was removed and then the buffer
  //would go from full to empty - with no data removed
  if(numbytes<1)
    return 0;

  //Return with error if we are attempting to read more data than is available
  if(numbytes > bytes_in_buffer(buffer))
    return -1;
  
  //If the buffer was full update the stop_offset
  if(buffer->stop_offset==-1)
    buffer->stop_offset=buffer->start_offset;

  //Update the start_offset
  buffer->start_offset = (buffer->start_offset+numbytes)&buffer->mask;
  
  return 0;
}

